import { User } from "../../models/user.model.js";
import { ApiError } from "../../utils/ApiError.js";
import { ApiResponse } from "../../utils/ApiResponse.js";
import jwt from "jsonwebtoken";
import { asyncHandler } from "../../utils/asyncHandler.js";
import { Token } from "../../models/token.model.js";
import { generateAccessAndRefreshTokens } from "../../utils/GenerateTokens.js";
import { UserAuth } from "../../models/userAuth.model.js";
import { sendOTP } from "../../services/emailService.js";
import { generateOTP } from "../../utils/generateOTP.js";


export const googleAuth = asyncHandler(async (req, res) => {

  const { idToken, userType } = req.body;
  if (!idToken || !userType) {
    throw new ApiError(400, "idToken (accessToken) and userType are required");
  }
 
  // ✅ Step 1: Get Google user info via access token
  const googleRes = await fetch("https://www.googleapis.com/oauth2/v3/userinfo", {

    headers: {
      Authorization: `Bearer ${idToken}`,
    },
  });
  if (!googleRes.ok) {
    throw new ApiError(401, "Invalid Google access token");
  }
  const payload = await googleRes.json();
  const { email, name: fullName, sub: providerId } = payload;
 
  // Step 2: Check if user exists
  // let user = await Model.findOne({ email });

   const user = await User.findOne({ email });

 
  if (user) {

    // 3. ✅ Check if userType matches
    if (user.userType !== userType) {
      throw new ApiError(
        403,
        `You are registered as a '${user.userType}'. Please use the correct login portal.`
      );
    }

    const userAuth = await UserAuth.findOne({ userId: user._id, provider: "google", providerId });

    if (!userAuth) {
      userAuth = await UserAuth.create({
      provider: "google",
      providerId,
      userId: user._id
    });
    }

    const verifiedTokenDoc = await Token.findOne({
      userId: user._id,
      tokenType: "verification",
      isVerified: true,
    });
  
    if (!verifiedTokenDoc) {

            const otp = generateOTP();
      const otpExpireAt = new Date(Date.now() + 10 * 60 * 1000); // 10 mins
      const verifyToken = jwt.sign(
        { userId: user._id },
        process.env.VERIFY_TOKEN_SECRET,
        { expiresIn: "15m" }
      );

      // ❌ Mark previous unverified tokens as used
      await Token.updateMany(
        { userId: user._id, tokenType: "verification", isVerified: false, isUsed: false },
        { $set: { isUsed: true } }
      );

      // 🔁 Upsert verification token
      let tokenDoc = await Token.findOne({
        userId: user._id,
        tokenType: "verification",
        isVerified: false,
        isUsed: false
      });

      if (tokenDoc) {
        tokenDoc.token = verifyToken;
        tokenDoc.otp = otp;
        tokenDoc.otpExpireAt = otpExpireAt;
        tokenDoc.expiresAt = new Date(Date.now() + 15 * 60 * 1000);
        tokenDoc.isVerified = false;
        tokenDoc.isUsed = false;
        tokenDoc.otpType = "phone";
        tokenDoc.authId = userAuth._id;
        await tokenDoc.save();
      } else {
        await Token.create({
          tokenType: "verification",
          token: verifyToken,
          otp,
          otpType: "phone",
          otpExpireAt,
          userId: user._id,
          authId: userAuth._id,
          expiresAt: new Date(Date.now() + 15 * 60 * 1000),
          isVerified: false,
          isUsed: false,
        });
      }

      // ✅ Send OTP to phone
      await sendOTP(user.email, otp);

        const options = {
          httpOnly: true,
          secure: process.env.NODE_ENV === "production",
          sameSite: "Lax",
          maxAge: 15 * 60 * 1000,
        };
      // ✅ Return partial info with verification token
      
      return res
      .cookie("verify_token", verifyToken, options)
      .setHeader("verify_token", verifyToken)
      .status(200)
      .json(
        new ApiResponse(
          200,
          {
            // requiresOTP: true,
              fullName: user.fullName,
              email: user.email,
              phone: user.phone,
              userType: user.userType,
              isNewUser: false,
              isVerified: false,
          },
          "User not verified. OTP sent to registered phone."
        )
      );
    }

    // ✅ Login flow
    const { accessToken, refreshToken } = await generateAccessAndRefreshTokens(user._id, null, {
      authId: userAuth._id,
      ipAddress: req.ip,
      userAgent: req.headers["user-agent"],
    });
 
    const options = {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: "Lax",
      maxAge: 7 * 24 * 60 * 60 * 1000,
    };

    return res
      .cookie("accessToken", accessToken, options)
      .cookie("refreshToken", refreshToken, options)
      .setHeader("accessToken", accessToken)
      .setHeader("refreshToken", refreshToken)
      .status(200)
      .json(
        new ApiResponse(
          200,
          {
            id: user._id,
            fullName: user.fullName,
            email: user.email,
            userType,
            isNewUser: false,
            isVerified: true,
          },
          "Login successfully"
        )
      );
  }
 
  // Step 3: New user — send prefill
  return res.status(200).json(
    new ApiResponse(
      200,
      {
        fullName,
        email,
        userType,
        providerId,
        isNewUser: true,
        isVerified: false,
      },

      "Google user needs to complete registration"

    )

  );

});

 
