import mongoose, { Schema } from "mongoose";
import bcrypt from "bcrypt";

const userAuthSchema = new Schema(
  {
    provider: {
      type: String,
      enum: ["local", "google", "facebook"],
      required: true
    },
    providerId: {
      type: String,
      required: function () {
        return this.provider !== "local";
      }
    },
    password: {
      type: String,
      required: function () {
        return this.provider === "local";
      }
    },
    isActive: {
      type: Boolean,
      default: true
    },
    lastLoginAt: {
      type: Date
    },
    userId: {
      type: Schema.Types.ObjectId,
      ref: "User",
      required: true
    }
  },
  {
    timestamps: { createdAt: true, updatedAt: false }
  }
);

// ✅ Pre-save hook to hash password for local accounts
userAuthSchema.pre("save", async function (next) {
  if (this.provider !== "local") return next();          // only hash for local
  if (!this.isModified("password")) return next();       // only if changed

  this.password = await bcrypt.hash(this.password, 10);
  next();
});

userAuthSchema.methods.isPasswordCorrect = async function (password) {
  return await bcrypt.compare(password, this.password);
};

userAuthSchema.index({ provider: 1, providerId: 1 }, { unique: true });

export const UserAuth = mongoose.model("UserAuth", userAuthSchema);
